/**
  Copyright (c) 2010 Freescale Semiconductor
  
  \file  	  Display.c
  \brief	  This is the Display driver File
  \brief	  Provides functionality control the display layers
  \author	  Freescale Semiconductor
  \author	  Automotive Systems Solutions Engineering
  \author	  IM, b06623
  \version	  2.0
  \revision	  $Revision: 21 $
  \date  	  $Date: 2010-09-09 19:26:09 -0500 (Thu, 09 Sep 2010) $ 
  
  * History:  11/May/2009 - Initial Version
  			  25/May/2010 - Update Version, support to DCU lite and DC3. DCU lite is now handled
  			  				with context switching.
  			  20/Abr/2010 - Supports RLE and Tile initialization.
  			  25/Ago/2010 - Added Display_UpdateClutDMA_cb to substitute Display_UpdateClutDMA_e -> deprecated					

* Copyright (c) 2010, Freescale, Inc.  All rights reserved.

*	MISRA VIOLATIONS:
	- [ MISRA 11.2 ]
	- [ MISRA 16.9 ]

* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
   
*/

#include 	"Display.h"

/* Driver Variables */
static void Display_HWCursorCallback(uint8_t channel);
uint8_t	Display_dcu = 1;

/**
* \brief	Display_InitLayer - Initializes the layer with the indicated graphic object
* \brief	This function enables the layer once all the data is entered. The CLUT Memory is not modified.
* \brief	Chroma and Alpha modes for the layer should be modifed independently
* \author	IM, b06623
* \param	DCU_Plane_t layer: the layer to be initialized.
* \param	const Graphics_Object_t* g_object: Is the graphic object info to initialize the layer
* \param	uint16_t x: X coordinate as start position.
* \param	uint16_t y: Y coordinate as start position.
* \return	void
* \todo
*/
void Display_InitLayer(uint8_t layer, Graphics_Object_t* g_object, uint16_t x, uint16_t y)
{
    Display_dcu = 1;
    if(Display_dcu == 1)
    {
    	DCU_LayerSetWidth( layer,  g_object->width);
    	DCU_LayerSetHeight( layer, g_object->height);
    	DCU_LayerSetX( layer, x );
    	DCU_LayerSetY( layer, y );

    	DCU_LayerOffset( layer ) = g_object->CLUTOffset;
    	DCU_LayerBPP( layer ) = g_object->BPP;

    	DCU_SetChroma(layer, g_object->initialChromaMax, g_object->initialChromaMin);
    	DCU_LayerAlpha( layer ) = g_object->initialAlpha;

    	DCU_LayerAddress( layer ) = g_object->address;
    	DCU_TileMode(layer, 0, 0, 0);

		if(g_object->coding==GRAPHICS_CODING_RLEINT)
		{
			DCU_RLE(layer,1,Graphics_GetCompSize(g_object->ext));	
		}		
		else		

    	if(g_object->coding==GRAPHICS_CODING_TILE)
    	{
    		DCU_TileMode(layer, 1, Graphics_GetTileW(g_object->ext), Graphics_GetTileH(g_object->ext));
    	}   	    	  	
    	DCU_LayerEnable(layer); 	
    }
}



/**
* \brief	Display_HWCursorCallback - Internal callback function that unitializes the DMA channel 
* \brief	used to update the cursor memory.
* \author	IM, b06623
* \param	uint8_t channel: Parameter received when in the DMA ISR. Tells which channel to unitialize.
* \return	void
* \todo
*/
static void Display_HWCursorCallback(uint8_t channel)
{
    DMA_UnInitChannel(channel);
    DCU_HWCursorEnable();
}

/**
* \brief	Display_UpdateClut - Updates the DCU Clut using the indicated Graphic object
* \brief	The CLUT must not be null and this function uses the CPU. The CLUT should be only updated
* \brief	when the DCU is off or during the VBLANK period.
* \author	IM, b06623
* \param	const Graphics_Object_t* g_object: Is the graphic object info to initialize the layer
* \return	void
* \todo
*/
void Display_UpdateClut(Graphics_Object_t *g_object)
{
    uint16_t i;
    uint16_t offset;

    offset = g_object->CLUTOffset;

    if(g_object->CLUT != NULL_PTR)
    {
		for(i = 0u; i < g_object->CLUTsize; i++)
		{
		    DCU_Clut( offset ) = g_object->CLUT[i];
		    offset++;
		}
    }
}

/**
* \brief	Display_UpdateClutDMA - Updates the DCU Clut using the indicated Graphic object
* \brief	The CLUT must not be null and this function uses the DMA. Origin CLUT must be aligned!!!!
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be initialized.
* \param	const Graphics_Object_t* g_object: Is the graphic object info to initialize the layer
* \return	Display_Error_t, possible resutls: DISPLAY_ERROR_DMA, DISPLAY_ERROR_OK
* \todo
*/
Display_Error_t Display_UpdateClutDMA(uint8_t eDMAChannel, Graphics_Object_t *g_object)
{
    DMA_ErrorType		error;
    Display_Error_t 	error1;

    error1 = DISPLAY_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );

    if((error != DMA_ERROR_OK) || (g_object->CLUT == NULL_PTR))
    {
		error1 = DISPLAY_ERROR_DMA;
    }
    else
    { 

		/* VIOLATION to [MISRA 11.2] Rationale: is passed to HW registers as uint32_t data, */ 
		/* pointers must be casted to that type */
		/* Source must be aligned */
		DMA_DestinationAddress(eDMAChannel) = (uint32_t)(&DCU_Clut(g_object->CLUTOffset));
		DMA_SourceAddress(eDMAChannel) 		= (uint32_t)(g_object->CLUT);
		DMA_SourceSize(eDMAChannel) 		= 2; 
		DMA_DestinationSize(eDMAChannel) 	= 2; 
		DMA_DestinationOffset(eDMAChannel) 	= 4;
		DMA_SourceOffset(eDMAChannel) 		= 4;
		DMA_MinorLoopCount(eDMAChannel)		= (uint32_t)g_object->CLUTsize*(uint32_t)4u;		
		DMA_SetMajorLoopCount(eDMAChannel,1);
	    /* VIOLATION TO [ MISRA 16.9 ] Rationale : There is no work-around to configure a run-time callback */  
		DMA_SetCallback(eDMAChannel, DMA_UnInitChannel);
		DMA_SetReady(eDMAChannel);			
    }

    return error1;
}


Display_CallbackType Display_CLUTCallback[DMA_MAX_DMACHANNELS];

void static Display_ClutDMA_CB(uint8_t ch)
{
	DMA_UnInitChannel(ch);
	if(Display_CLUTCallback[ch] != NULL_PTR)
	{
		Display_CLUTCallback[ch]();	
	}
}

Display_Error_t Display_UpdateClutDMA_cb(uint8_t eDMAChannel, Graphics_Object_t *g_object, Display_CallbackType cb)
{
    DMA_ErrorType		error;
    Display_Error_t 	error1;

    error1 = DISPLAY_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );

    if((error != DMA_ERROR_OK) || (g_object->CLUT == NULL_PTR))
    {
		error1 = DISPLAY_ERROR_DMA;
    }
    else
    { 

		/* VIOLATION to [MISRA 11.2] Rationale: is passed to HW registers as uint32_t data, */ 
		/* pointers must be casted to that type */
		/* Source must be aligned */
		DMA_DestinationAddress(eDMAChannel) = (uint32_t)(&DCU_Clut(g_object->CLUTOffset));
		DMA_SourceAddress(eDMAChannel) 		= (uint32_t)(g_object->CLUT);
		DMA_SourceSize(eDMAChannel) 		= 2; 
		DMA_DestinationSize(eDMAChannel) 	= 2; 
		DMA_DestinationOffset(eDMAChannel) 	= 4;
		DMA_SourceOffset(eDMAChannel) 		= 4;
		DMA_MinorLoopCount(eDMAChannel)		= (uint32_t)g_object->CLUTsize*(uint32_t)4u;		
		DMA_SetMajorLoopCount(eDMAChannel,1);
	    /* VIOLATION TO [ MISRA 16.9 ] Rationale : There is no work-around to configure a run-time callback */
	    Display_CLUTCallback[eDMAChannel] = cb;  
		DMA_SetCallback(eDMAChannel, Display_ClutDMA_CB);
		DMA_SetReady(eDMAChannel);			
    }

    return error1;
}


/* DEPRECATED, will be removed in future releases */
#ifndef REMOVE_DEPRECATED
uint8_t 			Display_ClutArraySize;
uint8_t				Display_ClutCtx;
Graphics_Object_t **Display_g_objectArray;

static void	Display_ClutDMACB(uint8_t eDMAChannel)
{
	DMA_UnInitChannel(eDMAChannel);
	Display_ClutArraySize--;
	if(Display_ClutArraySize != 0)
	{
		Display_UpdateClutDMA_e(eDMAChannel,Display_g_objectArray,(uint8_t)(0x80u|Display_ClutArraySize));	
	}
}

/**
* \brief	Display_UpdateClutDMA_e - Updates the DCU Clut using several Graphic object
* \brief	The CLUT must not be null and this function uses the DMA. Origin CLUT must be aligned!!!!
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be initialized.
* \param	const Graphics_Object_t** g_objectArray: Is the graphic object array info to initialize the layer
* \param	uint8_t size: the number of objects to process max 127
* \return	Display_Error_t, possible resutls: DISPLAY_ERROR_DMA, DISPLAY_ERROR_OK
* \todo
*/
Display_Error_t Display_UpdateClutDMA_e(uint8_t eDMAChannel, Graphics_Object_t *g_objectArray[], uint8_t size)
{
    DMA_ErrorType		error;
    Display_Error_t 	error1;
    Graphics_Object_t*	g_object;
    uint8_t				triggerFlag;

    error1 = DISPLAY_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );

	if(size & 0x80u)
	{
		triggerFlag = 1;
		size = (uint8_t)(size & 0x7Fu);	
	}
	else
	{
		triggerFlag = 0;
	}
	
	if(size != 0)
	{
		
		g_object = g_objectArray[size - 1];
		
    	if((error != DMA_ERROR_OK) || (g_object->CLUT == NULL_PTR))
    	{
			error1 = DISPLAY_ERROR_DMA;
    	}
    	else
   	 	{
   	 		Display_g_objectArray = g_objectArray;	
   	 		Display_ClutArraySize = size;		
						
			/* VIOLATION to [MISRA 11.2] Rationale: is passed to HW registers as uint32_t data, */ 
			/* pointers must be casted to that type */
			/* Source must be aligned */
			DMA_DestinationAddress(eDMAChannel) = (uint32_t)(&(DCU.CLUT[(g_object->CLUTOffset)]));				
			DMA_SourceAddress(eDMAChannel) 		= (uint32_t)(g_object->CLUT);
			DMA_SourceSize(eDMAChannel) 		= 2; 
			DMA_DestinationSize(eDMAChannel) 	= 2; 
			DMA_DestinationOffset(eDMAChannel) 	= 4;
			DMA_SourceOffset(eDMAChannel) 		= 4;			
			DMA_MinorLoopCount(eDMAChannel)		= (uint32_t)g_object->CLUTsize*(uint32_t)4u;		
			DMA_SetMajorLoopCount(eDMAChannel,1);
		    /* VIOLATION TO [ MISRA 16.9 ] Rationale : There is no work-around to configure a run-time callback */  
			DMA_SetCallback(eDMAChannel, Display_ClutDMACB);
			DMA_SetReady(eDMAChannel);				
			if(triggerFlag == 1)
	    	{
	    		DMA_Start(eDMAChannel);	
	    	}
				
    	}
    
	}
	else
	{
		error1 = DISPLAY_ERROR_SIZE;	
	}
	return error1;
}
#endif